import { useState } from 'react'
import { BookOpen, Calendar, CheckCircle, Clock, Users, Package, Award, UtensilsCrossed, ChefHat, FileText, Users as Community } from 'lucide-react'
import './BookClass.css'

const topics = [
  { id: 'baking', name: 'Выпечка', icon: UtensilsCrossed, color: '#FF6B35' },
  { id: 'desserts', name: 'Десерты', icon: ChefHat, color: '#F7931E' },
  { id: 'asian', name: 'Азиатская кухня', icon: UtensilsCrossed, color: '#FFB84D' },
  { id: 'vegan', name: 'Веганство', icon: ChefHat, color: '#8B4513' }
]

function BookClass() {
  const [selectedTopic, setSelectedTopic] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedTopic && selectedDate) {
      const newParticipation = {
        id: Date.now(),
        title: topics.find(t => t.id === selectedTopic)?.name || '',
        date: selectedDate,
        rating: null
      }
      
      const saved = localStorage.getItem('cookingParticipations')
      const participations = saved ? JSON.parse(saved) : { past: [] }
      participations.past.unshift(newParticipation)
      localStorage.setItem('cookingParticipations', JSON.stringify(participations))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedTopic('')
        setSelectedDate('')
      }, 3000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-class">
      <div className="container">
        <section className="page-intro">
          <div className="intro-content">
            <div className="intro-icon-wrapper">
              <BookOpen className="intro-main-icon" size={64} />
            </div>
            <h1 className="intro-title">Записаться на мастер-класс</h1>
            <p className="intro-description">
              Выберите интересующую вас тему и удобную дату. Погрузитесь в мир кулинарии вместе с нашими мастерами.
            </p>
          </div>
          <div className="intro-decorations">
            <div className="decoration decoration-1"></div>
            <div className="decoration decoration-2"></div>
            <div className="decoration decoration-3"></div>
          </div>
        </section>

        <section className="booking-section">
          <form className="booking-form" onSubmit={handleSubmit}>
            <div className="form-group">
              <label className="form-label">
                <FileText className="label-icon" size={20} />
                Выберите тему мастер-класса
              </label>
              <div className="topics-grid">
                {topics.map(topic => {
                  const IconComponent = topic.icon
                  return (
                    <button
                      key={topic.id}
                      type="button"
                      className={`topic-card ${selectedTopic === topic.id ? 'selected' : ''}`}
                      onClick={() => setSelectedTopic(topic.id)}
                      style={{
                        borderColor: selectedTopic === topic.id ? topic.color : 'rgba(255, 107, 53, 0.2)',
                        backgroundColor: selectedTopic === topic.id ? `${topic.color}15` : 'rgba(255, 255, 255, 0.9)'
                      }}
                    >
                      <IconComponent className="topic-icon" size={32} />
                      <span className="topic-name">{topic.name}</span>
                    </button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar className="label-icon" size={20} />
                Выберите дату
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <button type="submit" className="submit-btn" disabled={!selectedTopic || !selectedDate}>
              <CheckCircle className="btn-icon" size={20} />
              Записаться
            </button>

            {isSubmitted && (
              <div className="success-message">
                <CheckCircle className="success-icon" size={24} />
                <span>Вы успешно записались на мастер-класс!</span>
              </div>
            )}
          </form>
        </section>

        <section className="format-section">
          <h2 className="section-title">
            <Clock className="section-icon" size={24} />
            Формат занятий
          </h2>
          <div className="format-cards-vertical">
            <div className="format-card-full">
              <div className="format-card-content">
                <div className="format-icon-wrapper">
                  <Clock className="format-icon" size={48} />
                </div>
                <div className="format-text">
                  <h3>Длительность и интенсивность</h3>
                  <p>Каждое занятие длится 3 часа — этого времени достаточно, чтобы полностью освоить все техники и приготовить блюдо от начала до конца. Вы получите не только практические навыки, но и глубокое понимание кулинарных процессов. Наши мастера уделяют внимание каждому участнику, объясняя тонкости работы с ингредиентами и профессиональными инструментами.</p>
                </div>
              </div>
            </div>
            <div className="format-card-full">
              <div className="format-card-content">
                <div className="format-icon-wrapper">
                  <Users className="format-icon" size={48} />
                </div>
                <div className="format-text">
                  <h3>Малые группы и индивидуальный подход</h3>
                  <p>Мы формируем группы до 8 человек, что обеспечивает максимальное внимание каждому участнику. Вы сможете задавать вопросы, получать персональные рекомендации и работать в комфортной атмосфере. Наши шеф-повара следят за техникой каждого участника и помогают исправить ошибки в процессе приготовления.</p>
                </div>
              </div>
            </div>
            <div className="format-card-full">
              <div className="format-card-content">
                <div className="format-icon-wrapper">
                  <Package className="format-icon" size={48} />
                </div>
                <div className="format-text">
                  <h3>Результат и материалы</h3>
                  <p>Все приготовленные блюда вы забираете с собой в красивой упаковке. Дополнительно вы получите подробные рецепты с пошаговыми инструкциями, список необходимых ингредиентов и рекомендации по выбору продуктов. Также мы предоставляем доступ к онлайн-библиотеке с видео-уроками и дополнительными материалами по теме мастер-класса.</p>
                </div>
              </div>
            </div>
          </div>
        </section>

        <section className="benefits-section">
          <h2 className="section-title">
            <Award className="section-icon" size={24} />
            Что вы получите
          </h2>
          <div className="benefits-cards-vertical">
            <div className="benefit-card-full">
              <div className="benefit-card-content">
                <div className="benefit-icon-wrapper">
                  <Award className="benefit-icon" size={48} />
                </div>
                <div className="benefit-text">
                  <h3>Профессиональные навыки и техники</h3>
                  <p>Вы освоите профессиональные кулинарные техники, которые используют шеф-повара в ресторанах. Научитесь правильно работать с ножом, понимать температуру приготовления, создавать идеальные текстуры и балансировать вкусы. Эти навыки останутся с вами на всю жизнь и помогут готовить ресторанные блюда в домашних условиях.</p>
                </div>
              </div>
            </div>
            <div className="benefit-card-full">
              <div className="benefit-card-content">
                <div className="benefit-icon-wrapper">
                  <BookOpen className="benefit-icon" size={48} />
                </div>
                <div className="benefit-text">
                  <h3>Эксклюзивные рецепты и секреты</h3>
                  <p>Получите доступ к уникальным рецептам, которые передаются от мастера к мастеру. Узнайте секреты приготовления, которые не найти в обычных кулинарных книгах — от правильного замеса теста до создания идеального соуса. Наши шеф-повара делятся проверенными годами техниками и профессиональными хитростями.</p>
                </div>
              </div>
            </div>
            <div className="benefit-card-full">
              <div className="benefit-card-content">
                <div className="benefit-icon-wrapper">
                  <Community className="benefit-icon" size={48} />
                </div>
                <div className="benefit-text">
                  <h3>Сообщество единомышленников</h3>
                  <p>Присоединитесь к сообществу любителей кулинарии, где вы сможете обмениваться опытом, делиться своими успехами и получать поддержку. После мастер-класса вы получите доступ к закрытой группе, где участники делятся фотографиями своих блюд, задают вопросы мастерам и получают дополнительные советы по приготовлению.</p>
                </div>
              </div>
            </div>
          </div>
        </section>
      </div>
    </div>
  )
}

export default BookClass

